<?php
/**
 * Manager blocks
 */
class WC_Vendors_Pro_Blocks {

    /**
     * Blocks
     *
     * @var array $blocks the blocks
     */
    private $blocks = array();

    /**
     * Constructor
     */
    public function __construct() {
        $this->include_files();
        $this->blocks = array(
            'store-total-sales'       => 'wcv_render_block_store_total_sales',
            'store-contact'           => 'wcv_render_block_store_contact',
            'store-categories'        => 'wcv_render_block_store_categories',
            'store-search'            => 'wcv_render_block_store_search',
            'store-recent-products'   => 'wcv_render_block_store_recent_products',
            'store-short-description' => 'wcv_render_block_store_short_description',
            'store-ratings'           => 'wcv_render_block_store_ratings',
            'store-social-media'      => 'wcv_render_block_store_social_media',
            'store-map-address'       => 'wcv_render_block_store_map_address',
            'store-price-filter'      => 'wcv_render_block_store_price_filter',
            'store-attributes-filter' => 'wcv_render_block_store_attributes_filter',
            'store-ratings-filter'    => 'wcv_render_block_store_ratings_filter',
        );
        add_action( 'init', array( $this, 'load_blocks' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_extras_access' ) );
        add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
        add_action( 'pre_get_posts', array( $this, 'pre_get_posts' ) );
    }

    /**
     * Add query vars
     *
     * @param array $vars query vars.
     *
     * @return array
     */
    public function add_query_vars( $vars ) {
        // Gget all product attributes.
        $attributes = wc_get_attribute_taxonomies();
        foreach ( $attributes as $attribute ) {
            $attr_slug = wc_get_attribute( $attribute->attribute_id );
            $vars[]    = 'filter_' . $attr_slug->slug;
        }
        $vars[] = 'filter_attribute';
        return $vars;
    }

    /**
     * Preget posts
     *
     * @param WP_Query $query query.
     *
     * @return array $query query.
     */
    public function pre_get_posts( $query ) {
        if ( ! is_admin() && $query->is_main_query() ) {
            $attributes = $query->get( 'filter_attribute' );
            if ( empty( $attributes ) ) {
                return $query;
            }
            $attributes = explode( ',', $attributes );

            foreach ( $attributes as $attribute ) {
                $terms = $query->get( 'filter_' . $attribute );
                if ( empty( $terms ) ) {
                    continue;
                }
                $query->set(
                    'tax_query',
                    array(
                        'relation' => 'AND',
                        array(
                            'taxonomy' => $attribute,
                            'field'    => 'term_id',
                            'terms'    => $terms,
                        ),
                    )
                );
            }
        }
        return $query;
    }


    /**
     * Enqueue extras access
     */
    public function enqueue_extras_access() {
        if ( ! is_admin() ) {
            wp_enqueue_style( 'jquery-ui-style', WC()->plugin_url() . '/assets/css/jquery-ui/jquery-ui.min.css', array(), WC()->version );
        }
    }

    /**
     * Include functions and callbacks
     */
    private function include_files() {
        require_once __DIR__ . '/wcvendors-blocks-functions.php';
        require_once __DIR__ . '/wcvendors-blocks-callbacks.php';
    }

    /**
     * Load blocks
     *
     * @return void
     *
     * @throws Exception If block does not exist.
     *
     * @version 1.8.7
     * @since   1.8.7
     */
    public function load_blocks() {
        $blocks = $this->get_blocks();
        foreach ( $blocks as $dir => $render_callback ) {
            if ( ! file_exists( __DIR__ . '/build/' . $dir . '/block.json' ) ) {
                throw new Exception( 'Block ' . $dir . ' does not exist.' );
            }

            $args = array();
            if ( ! empty( $render_callback ) && is_callable( $render_callback ) ) {
                $args['render_callback'] = $render_callback;
            }
            register_block_type( __DIR__ . '/build/' . $dir, $args );
        }
    }

    /**
     * Get blocks
     *
     * @return array
     */
    private function get_blocks() {
        return $this->blocks;
    }
}
new WC_Vendors_Pro_Blocks();
